-- -------------------------------------------------------------------------- --
-- EnhancedStackSplit by kunda                                                --
-- -------------------------------------------------------------------------- --
--                                                                            --
-- Features:                                                                  --
-- - 1-Click Mode:                                                            --
--   -> 'M' Button                                                            --
--      Toggle: '1-Click Mode' / 'Original WoW Mode' [DEFAULT]                --
--   Split a stack with just one click when you click on one of the numbered  --
--   Buttons (OK(1),2,3,4,5,6,7,8,9,10 and 20 - XL: 12,14,15,16 and 100).     --
--   Split is placed on the cursor!!! so you can place it in a slot, trade,   --
--   destroy or what ever.                                                    --
--                                                                            --
-- - Auto Split Mode:                                                         --
--   -> 'Auto' Button                                                         --
--   Automatically splits a stack and place the split item(s) in a free bag   --
--   slot when you click on one of the numbered Buttons. Free bag slots       --
--   depends on ItemFamily from item you want to split and on ItemFamily from --
--   bag(s) with free slots. Items are placed from top left to bottom right   --
--   slot in a bag.                                                           --
--   Example: Free bagslots = 20                                              --
--            Item with StackSize = 137                                       --
--            - you click on 5                                                --
--              you get: 20 items with 5 Itemstacksize in your bag(s)         --
--                       leftover = 37                                        --
--                       (20 because free bagslots = 20)                      --
--            - you click on 10                                               --
--              you get: 13 items with 10 Itemstacksize in your bag(s)        --
--                       leftover = 7                                         --
--            leftover and last n-stacksize remains in the original bagslot   --
--   Note: Auto-split items are only placed in the five normal bags including --
--         special bags (or less if you have not five bags).                  --
--         The 'Auto Split Mode' should be only accessible via bags and bank, --
--         but I did not find a good way to do this. So it's possible that    --
--         this mode is available where it shouldn't be. If you try to auto   --
--         split in such cases the SplitFrame can stuck. Just press ESC to    --
--         close it. Bag and Bank Addons MUST provide the bag/slot IDs in the --
--         same way the original UI does, otherwise EnhancedStackSplit can    --
--         not auto-split such stacks or you can get the stuck-error I        --
--         described above.                                                   --
--         Most bag/bank addons works perfect with EnhancedStackSplit because --
--         the addon authors made a good job. :)                              --
--                                                                            --
-- -------------------------------------------------------------------------- --
--                                                                            --
-- Credits:                                                                   --
-- LockSlot functions from ClosetGnome by Rabbit.                             --
--                                                                            --
-- NOTE:                                                                      --
-- You can shift-click on any stackable item, press a number on your keyboard --
-- and click OK or press ENTER to split a stack!!!                            --
--                                                                            --
-- -------------------------------------------------------------------------- --

-- ---------------------------------------------------------------------------------------------------------------------
EnhancedStackSplit_Options = {}                 -- SavedVariable table

local EnhancedStackSplit = CreateFrame("Frame") -- container
local L = EnhancedStackSplit_Locales            -- localization table

local _G = _G
local autoSplitMode = false
local splitItemBag
local splitItemSlot
local maxStackSize = 0
local slotLocks = nil
local autoSplitStackCount
local autoSplitNumStacks
local autoSplitNumber
local autoSplitLeftOver
local autoSplitItemLink

local PositionAnchor
local PositionParent
local PositionAnchorTo

local splitMode = {
	[1] = L["Original WoW Mode"],
	[2] = L["1-Click Mode"],
	[3] = L["Auto Split Mode"]
}
local blockedFrames = {
	[1] = "MerchantItem",
	[2] = "GuildBank"
}
local mainBankFrames = {
	[1] = "BankFrame", -- default Blizzard Bank Frame
	[2] = "TBnkFrame"  -- TBag
}
-- ---------------------------------------------------------------------------------------------------------------------

-- Helper functions START ----------------------------------------------------------------------------------------------
local function reclaim(t) -- LockSlot function
	if type(t) ~= "table" then return end
	for k, v in pairs(t) do
		if type(v) == "table" then
			t[k] = reclaim(v)
		else
			t[k] = nil
		end
	end
	t = nil
	return nil
end

local function LockSlot(bag, slot) -- LockSlot function
	if not slotLocks then slotLocks = {} end
	if not slotLocks[bag] then slotLocks[bag] = {} end
	slotLocks[bag][slot] = true
end

local function IsSlotLocked(bag, slot) -- LockSlot function
	if slotLocks and slotLocks[bag] and slotLocks[bag][slot] then
		return true
	end
	return false
end

local function ClearSlotLocks() -- LockSlot function
	slotLocks = reclaim(slotLocks)
end

local function IsBag(bag)
	local invID = ContainerIDToInventoryID(bag)
	local bagLink = GetInventoryItemLink("player", invID)
	if bagLink then
		return bagLink
	else
		return false
	end
end

local function GetFreeBagSlots()
	local freeSlots = GetContainerNumFreeSlots(0)
	local containerBagType = 0
	local itemBagType = GetItemFamily(autoSplitItemLink)
	for bag = 1, NUM_BAG_SLOTS do
		local bagLink = IsBag(bag)
		if bagLink then
			containerBagType = GetItemFamily(bagLink)
			if containerBagType == 0 or containerBagType == itemBagType then
				freeSlots = freeSlots + GetContainerNumFreeSlots(bag)
			end
		end
	end
	return freeSlots
end

local function GetFreeSlot()
	local containerBagType = 0
	local itemBagType = GetItemFamily(autoSplitItemLink)
	local goodBag = true
	for bag = 0, NUM_BAG_SLOTS do
		if bag > 0 then
			local bagLink = IsBag(bag)
			if bagLink then
				containerBagType = GetItemFamily(bagLink)
				if containerBagType == 0 or containerBagType == itemBagType then
					goodBag = true
				else
					goodBag = false
				end
			end
		end
		if goodBag then
			for slot = 1, GetContainerNumSlots(bag) do
				if not IsSlotLocked(bag, slot) then
					if not GetContainerItemLink(bag, slot) then
						return bag, slot
					end
				end
			end
		end
	end
	return nil
end

local function AutoSplitCalc(num)
	local freeSlots = GetFreeBagSlots()
	local numStacks = floor(maxStackSize/num)
	if numStacks > freeSlots then
		numStacks = freeSlots
	end
	local leftOver = maxStackSize-(numStacks*num)
	return freeSlots or 0, numStacks or 0, leftOver or 0
end

local function createButton(name, parent, template, anchorparent, width, anchorp, anchorrp, adimx, adimy, text, func1, func2)
	local b = CreateFrame("Button", name, _G[parent], template)
	b:SetWidth(width)
	b:SetHeight(24)
	b:SetPoint(anchorp, anchorparent, anchorrp, adimx, adimy)
	b:SetText(text)
	b:SetScript("OnClick", func1)
	if func2 then
		b:SetScript("OnEnter", func2)
		b:SetScript("OnLeave", function() EnhancedStackSplitTextAutoText2:SetText("") end)
	end
end
-- Helper functions END ------------------------------------------------------------------------------------------------

-- ---------------------------------------------------------------------------------------------------------------------
function EnhancedStackSplit:InitOptions()
	if EnhancedStackSplit_Options.version and EnhancedStackSplit_Options.version < 4 then
		EnhancedStackSplit_Options.chat = nil
		EnhancedStackSplit_Options.version = 4
	end
	if EnhancedStackSplit_Options.version == nil then EnhancedStackSplit_Options.version = 4 end
	if EnhancedStackSplit_Options.mode    == nil then EnhancedStackSplit_Options.mode    = 1 end
	if EnhancedStackSplit_Options.XLmode  == nil then EnhancedStackSplit_Options.XLmode  = false end
end

function EnhancedStackSplit:CreateFrames()
	local topframe = CreateFrame("Frame", "EnhancedStackSplitTopTextureFrame", StackSplitFrame)
	topframe:SetPoint("TOPLEFT", "StackSplitFrame", "TOPLEFT", 0, 2)
	topframe:SetWidth(172)
	topframe:SetHeight(20)
	local texture = topframe:CreateTexture(nil, "BACKGROUND")
	texture:SetTexture("Interface\\MoneyFrame\\UI-MoneyFrame2")
	texture:SetAllPoints(topframe)
	texture:SetBlendMode("ALPHAKEY")
	texture:SetTexCoord(0, 172/256, 0, 20/128)
	local text = topframe:CreateFontString("EnhancedStackSplitTextFrameTXT", "BACKGROUND", "GameFontNormalSmall")
	text:SetPoint("TOP", "EnhancedStackSplitTopTextureFrame", "TOP", 1, -8)
	text:SetJustifyH("CENTER")
	local text = topframe:CreateFontString("EnhancedStackSplitTextAutoText1", "BACKGROUND", "GameFontNormalSmall")
	text:SetPoint("TOP", "EnhancedStackSplitTextFrameTXT", "BOTTOM", 0, -2)
	text:SetJustifyH("CENTER")
	local text = topframe:CreateFontString("EnhancedStackSplitTextAutoText2", "BACKGROUND", "GameFontNormalSmall")
	text:SetPoint("TOP", "EnhancedStackSplitTextAutoText1", "BOTTOM", 0, -2)
	text:SetJustifyH("CENTER")

	local frame = CreateFrame("Frame", "EnhancedStackSplitBottomTextureFrame", StackSplitFrame)
	frame:EnableMouse(true)
	frame:SetPoint("TOPLEFT", "StackSplitFrame", "BOTTOMLEFT", 0, 27)
	frame:SetWidth(172)
	frame:SetHeight(30)
	local texture = frame:CreateTexture(nil, "BACKGROUND")
	texture:SetTexture("Interface\\MoneyFrame\\UI-MoneyFrame")
	texture:SetAllPoints(frame)
	texture:SetTexCoord(0, 172/256, 46/128, 76/128)

	local frame = CreateFrame("Frame", "EnhancedStackSplitBottom2TextureFrame", StackSplitFrame)
	frame:EnableMouse(true)
	frame:SetPoint("TOPLEFT", "EnhancedStackSplitBottomTextureFrame", "BOTTOMLEFT", 0, 0)
	frame:SetWidth(172)
	frame:SetHeight(50)
	local texture = frame:CreateTexture(nil, "BACKGROUND")
	texture:SetTexture("Interface\\MoneyFrame\\UI-MoneyFrame")
	texture:SetAllPoints(frame)
	texture:SetTexCoord(0, 172/256, 46/128, 96/128)

	local autoframe = CreateFrame("Frame", "EnhancedStackSplitAutoTextureFrame", StackSplitFrame)
	autoframe:SetPoint("TOPLEFT", "StackSplitFrame", "TOPLEFT", 16, -13)
	autoframe:SetWidth(142)
	autoframe:SetHeight(37)
	autoframe:Hide()
	local texture = autoframe:CreateTexture(nil, "BACKGROUND")
	texture:SetTexture("Interface\\MoneyFrame\\UI-MoneyFrame2")
	texture:SetAllPoints(autoframe)
	texture:SetTexCoord(16/256, 158/256, 13/128, 50/128)
	local texture = autoframe:CreateTexture(nil, "HIGH")
	texture:SetTexture("Interface\\DialogFrame\\DialogAlertIcon")
	texture:SetPoint("BOTTOM", "EnhancedStackSplitAutoTextureFrame", "TOPRIGHT", 6, 0)
	texture:SetWidth(41)
	texture:SetHeight(32)
	texture:SetTexCoord(11/64, 52/64, 16/64, 48/64)
	local texture = autoframe:CreateTexture(nil, "HIGH")
	texture:SetTexture("Interface\\DialogFrame\\DialogAlertIcon")
	texture:SetPoint("BOTTOM", "EnhancedStackSplitAutoTextureFrame", "TOPLEFT", -6, 0)
	texture:SetWidth(41)
	texture:SetHeight(32)
	texture:SetTexCoord(11/64, 52/64, 16/64, 48/64)

	createButton("EnhancedStackSplitAuto1Button", "StackSplitFrame", "UIPanelButtonTemplate", "StackSplitFrame", 64, "RIGHT", "BOTTOM", -3, 40, "1", function() EnhancedStackSplit:Split(1) end, function() EnhancedStackSplit:AutoSplitInfo(1) end)
	EnhancedStackSplitAuto1Button:Hide()

	createButton("EnhancedStackSplitButton1",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitBottomTextureFrame", 22, "TOPLEFT", "TOPLEFT",   10, 2,  "2", function() EnhancedStackSplit:Split(2) end,  function() EnhancedStackSplit:AutoSplitInfo(2) end)
	createButton("EnhancedStackSplitButton2",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton1",            22, "TOPLEFT", "TOPRIGHT",  -3, 0,  "3", function() EnhancedStackSplit:Split(3) end,  function() EnhancedStackSplit:AutoSplitInfo(3) end)
	createButton("EnhancedStackSplitButton3",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton2",            22, "TOPLEFT", "TOPRIGHT",  -3, 0,  "4", function() EnhancedStackSplit:Split(4) end,  function() EnhancedStackSplit:AutoSplitInfo(4) end)
	createButton("EnhancedStackSplitButton4",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton3",            22, "TOPLEFT", "TOPRIGHT",  -3, 0,  "5", function() EnhancedStackSplit:Split(5) end,  function() EnhancedStackSplit:AutoSplitInfo(5) end)
	createButton("EnhancedStackSplitButton5",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton4",            22, "TOPLEFT", "TOPRIGHT",  -3, 0,  "6", function() EnhancedStackSplit:Split(6) end,  function() EnhancedStackSplit:AutoSplitInfo(6) end)
	createButton("EnhancedStackSplitButton6",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton5",            22, "TOPLEFT", "TOPRIGHT",  -3, 0,  "7", function() EnhancedStackSplit:Split(7) end,  function() EnhancedStackSplit:AutoSplitInfo(7) end)
	createButton("EnhancedStackSplitButton7",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton6",            22, "TOPLEFT", "TOPRIGHT",  -3, 0,  "8", function() EnhancedStackSplit:Split(8) end,  function() EnhancedStackSplit:AutoSplitInfo(8) end)
	createButton("EnhancedStackSplitButton8",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton7",            22, "TOPLEFT", "TOPRIGHT",  -3, 0,  "9", function() EnhancedStackSplit:Split(9) end,  function() EnhancedStackSplit:AutoSplitInfo(9) end)
	createButton("EnhancedStackSplitButton9",  "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton1",            26, "TOPLEFT", "BOTTOMLEFT", 0, 2, "10", function() EnhancedStackSplit:Split(10) end, function() EnhancedStackSplit:AutoSplitInfo(10) end)
	createButton("EnhancedStackSplitButton10", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton9",            26, "TOPLEFT", "TOPRIGHT",  -3, 0, "20", function() EnhancedStackSplit:Split(20) end, function() EnhancedStackSplit:AutoSplitInfo(20) end)

	createButton("EnhancedStackSplitButton11", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton1",            26, "TOPLEFT", "BOTTOMLEFT", 7, 2.5, "10", function() EnhancedStackSplit:Split(10) end, function() EnhancedStackSplit:AutoSplitInfo(10) end)
	createButton("EnhancedStackSplitButton12", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton11",           26, "TOPLEFT", "TOPRIGHT",  -3, 0, "12", function() EnhancedStackSplit:Split(12) end, function() EnhancedStackSplit:AutoSplitInfo(12) end)
	createButton("EnhancedStackSplitButton13", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton12",           26, "TOPLEFT", "TOPRIGHT",  -3, 0, "14", function() EnhancedStackSplit:Split(14) end, function() EnhancedStackSplit:AutoSplitInfo(14) end)
	createButton("EnhancedStackSplitButton14", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton13",           26, "TOPLEFT", "TOPRIGHT",  -3, 0, "15", function() EnhancedStackSplit:Split(15) end, function() EnhancedStackSplit:AutoSplitInfo(15) end)
	createButton("EnhancedStackSplitButton15", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton14",           26, "TOPLEFT", "TOPRIGHT",  -3, 0, "16", function() EnhancedStackSplit:Split(16) end, function() EnhancedStackSplit:AutoSplitInfo(16) end)
	createButton("EnhancedStackSplitButton16", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton15",           26, "TOPLEFT", "TOPRIGHT",  -3, 0, "20", function() EnhancedStackSplit:Split(20) end, function() EnhancedStackSplit:AutoSplitInfo(20) end)

	createButton("EnhancedStackSplitAutoSplitButton", "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitButton8",         1, "TOPRIGHT", "BOTTOMRIGHT", 0, 2, L["Auto"], function() EnhancedStackSplit:ModeToggle(3) end)
	createButton("EnhancedStackSplitModeTXTButton",   "EnhancedStackSplitBottomTextureFrame", "UIPanelButtonTemplate2", "EnhancedStackSplitAutoSplitButton", 1, "TOPRIGHT", "TOPLEFT",     3, 0, L["M"],    function() EnhancedStackSplit:ModeToggle(1) end)
	EnhancedStackSplitAutoSplitButton:SetWidth(ceil(EnhancedStackSplitAutoSplitButton:GetTextWidth())+10)
	EnhancedStackSplitModeTXTButton:SetWidth(ceil(EnhancedStackSplitModeTXTButton:GetTextWidth())+10)

	local XLButton = CreateFrame("Button", "EnhancedStackSplitXLModeButton", EnhancedStackSplitBottomTextureFrame)
	XLButton:SetWidth(20)
	XLButton:SetHeight(12)
	XLButton:SetPoint("CENTER", "EnhancedStackSplitBottomTextureFrame", "CENTER", 0, -30)

	local textureNormal = XLButton:CreateTexture("EnhancedStackSplitXLModeButtonNormalTexture", "ARTWORK")
	textureNormal:SetAllPoints(XLButton)
	textureNormal:SetTexture("Interface\\Buttons\\UI-TotemBar")
	textureNormal:SetTexCoord(100/128, 126/128, 104/256, 119/256)
	XLButton:SetNormalTexture(textureNormal)
	local textureHighlight = XLButton:CreateTexture("EnhancedStackSplitXLModeButtonHighlightTexture", "ARTWORK")
	textureHighlight:SetPoint("TOPLEFT", 3, -3)
	textureHighlight:SetPoint("BOTTOMRIGHT", -3, 3)
	textureHighlight:SetTexture("Interface\\Buttons\\UI-TotemBar")
	textureHighlight:SetTexCoord(72/128, 92/128, 69/256, 79/256)
	XLButton:SetHighlightTexture(textureHighlight)
	local texturePushed = XLButton:CreateTexture("EnhancedStackSplitXLModeButtonPushedTexture", "ARTWORK")
	texturePushed:SetPoint("TOPLEFT", 1, -1)
	texturePushed:SetPoint("BOTTOMRIGHT", -1, 1)
	texturePushed:SetTexture("Interface\\Buttons\\UI-TotemBar")
	texturePushed:SetTexCoord(100/128, 126/128, 104/256, 119/256)
	XLButton:SetPushedTexture(texturePushed)
	local textureDisabled = XLButton:CreateTexture("EnhancedStackSplitXLModeButtonDisabledTexture", "ARTWORK")
	textureDisabled:SetAllPoints(XLButton)
	textureDisabled:SetTexture("Interface\\Buttons\\UI-TotemBar")
	textureDisabled:SetTexCoord(100/128, 126/128, 104/256, 119/256)
	XLButton:SetDisabledTexture(textureDisabled)

	local shaderSupported = textureDisabled:SetDesaturated(true)
	if not shaderSupported then
		textureDisabled:SetVertexColor(0.5, 0.5, 0.5)
	end

	XLButton:SetScript("OnClick",  function()
		EnhancedStackSplit:XLModeToggle()
	end)

	EnhancedStackSplit:ModeSettings(EnhancedStackSplit_Options.mode)
	EnhancedStackSplit:RepositionButtons()
end

function EnhancedStackSplit:PositionSplitFrame()
	if EnhancedStackSplit_Options.XLmode then
		StackSplitFrame:SetPoint(PositionAnchor, PositionParent, PositionAnchorTo, 0, 35)
	else
		StackSplitFrame:SetPoint(PositionAnchor, PositionParent, PositionAnchorTo, 0, 14)
	end
end

function EnhancedStackSplit:RepositionButtons()
	if EnhancedStackSplit_Options.XLmode then
		EnhancedStackSplitBottomTextureFrame:SetHeight(30)
		EnhancedStackSplitAutoSplitButton:SetPoint("TOPRIGHT", "EnhancedStackSplitButton8", "BOTTOMRIGHT", 0, -19)

		EnhancedStackSplitButton9:SetText("100")
		EnhancedStackSplitButton9:SetWidth(34)
		EnhancedStackSplitButton9:SetScript("OnClick", function() EnhancedStackSplit:Split(100) end)
		EnhancedStackSplitButton9:SetScript("OnEnter", function() EnhancedStackSplit:AutoSplitInfo(100) end)
		EnhancedStackSplitButton9:SetPoint("TOPLEFT", "EnhancedStackSplitButton1", "BOTTOMLEFT", 0, -19)
		EnhancedStackSplitButton10:Hide()

		EnhancedStackSplitButton11:Show()
		EnhancedStackSplitButton12:Show()
		EnhancedStackSplitButton13:Show()
		EnhancedStackSplitButton14:Show()
		EnhancedStackSplitButton15:Show()
		EnhancedStackSplitButton16:Show()

		EnhancedStackSplitXLModeButton:SetPoint("CENTER", "EnhancedStackSplitBottomTextureFrame", "CENTER", 0, -45)
		EnhancedStackSplitXLModeButtonNormalTexture:SetTexCoord(100/128, 126/128, 123/256, 138/256)
		EnhancedStackSplitXLModeButtonPushedTexture:SetTexCoord(100/128, 126/128, 123/256, 138/256)
		EnhancedStackSplitXLModeButtonDisabledTexture:SetTexCoord(100/128, 126/128, 123/256, 138/256)
		EnhancedStackSplitXLModeButtonHighlightTexture:SetTexCoord(72/128, 92/128, 88/256, 98/256)
	else
		EnhancedStackSplitBottomTextureFrame:SetHeight(9)
		EnhancedStackSplitAutoSplitButton:SetPoint("TOPRIGHT", "EnhancedStackSplitButton8", "BOTTOMRIGHT", 0, 2)

		EnhancedStackSplitButton9:SetText("10")
		EnhancedStackSplitButton9:SetWidth(26)
		EnhancedStackSplitButton9:SetScript("OnClick", function() EnhancedStackSplit:Split(10) end)
		EnhancedStackSplitButton9:SetScript("OnEnter", function() EnhancedStackSplit:AutoSplitInfo(10) end)
		EnhancedStackSplitButton9:SetPoint("TOPLEFT", "EnhancedStackSplitButton1", "BOTTOMLEFT", 0, 2)
		EnhancedStackSplitButton10:Show()

		EnhancedStackSplitButton11:Hide()
		EnhancedStackSplitButton12:Hide()
		EnhancedStackSplitButton13:Hide()
		EnhancedStackSplitButton14:Hide()
		EnhancedStackSplitButton15:Hide()
		EnhancedStackSplitButton16:Hide()

		EnhancedStackSplitXLModeButton:SetPoint("CENTER", "EnhancedStackSplitBottomTextureFrame", "CENTER", 0, -34)
		EnhancedStackSplitXLModeButtonNormalTexture:SetTexCoord(100/128, 126/128, 104/256, 119/256)
		EnhancedStackSplitXLModeButtonPushedTexture:SetTexCoord(100/128, 126/128, 104/256, 119/256)
		EnhancedStackSplitXLModeButtonDisabledTexture:SetTexCoord(100/128, 126/128, 104/256, 119/256)
		EnhancedStackSplitXLModeButtonHighlightTexture:SetTexCoord(72/128, 92/128, 69/256, 79/256)
	end
end

function EnhancedStackSplit:XLModeToggle()
	EnhancedStackSplit_Options.XLmode = not EnhancedStackSplit_Options.XLmode
	EnhancedStackSplit:PositionSplitFrame()
	EnhancedStackSplit:RepositionButtons()
	if autoSplitMode then
		EnhancedStackSplit:ModeSettings(3)
	else
		EnhancedStackSplit:ModeSettings(EnhancedStackSplit_Options.mode)
	end
end

function EnhancedStackSplit:ModeToggle(mode)
	if mode == 3 then
		autoSplitMode = true
		EnhancedStackSplit:ModeSettings(3)
	else
		if not autoSplitMode then
			if EnhancedStackSplit_Options.mode == 2 then
				EnhancedStackSplit_Options.mode = 1
			else
				EnhancedStackSplit_Options.mode = 2
			end
		end
		autoSplitMode = false
		EnhancedStackSplit:ModeSettings(EnhancedStackSplit_Options.mode)
	end
end

function EnhancedStackSplit:AutoSplitButtonToggle(toggle)
	if toggle then
		for i = 1, 16 do
			EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton"..i, 1)
		end
		EnhancedStackSplitXLModeButton:Enable()
		EnhancedStackSplitAuto1Button:Enable()
		EnhancedStackSplitModeTXTButton:Enable()
		StackSplitCancelButton:Enable()
	else
		for i = 1, 16 do
			EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton"..i, 0)
		end
		EnhancedStackSplitXLModeButton:Disable()
		EnhancedStackSplitAuto1Button:Disable()
		EnhancedStackSplitModeTXTButton:Disable()
		StackSplitCancelButton:Disable()
	end
end

function EnhancedStackSplit:ButtonTweak(button, state)
	-- button: string
	-- state : 0=disable | 1=enable
	if state == 0 then
		_G[button]:Disable()
		_G[button]:SetDisabledTexture(nil)
		_G[button.."Left"]:SetTexture("Interface\\Buttons\\UI-Panel-Button-Disabled")
		_G[button.."Middle"]:SetTexture("Interface\\Buttons\\UI-Panel-Button-Disabled")
		_G[button.."Right"]:SetTexture("Interface\\Buttons\\UI-Panel-Button-Disabled")
		_G[button]:SetScript("OnMouseDown", nil)
		_G[button]:SetScript("OnMouseUp", nil)
	else
		_G[button]:Enable()
		_G[button.."Left"]:SetTexture("Interface\\Buttons\\UI-Panel-Button-Up")
		_G[button.."Middle"]:SetTexture("Interface\\Buttons\\UI-Panel-Button-Up")
		_G[button.."Right"]:SetTexture("Interface\\Buttons\\UI-Panel-Button-Up")
	end
end

function EnhancedStackSplit:ModeSettings(mode)
	for i = 1, 16 do
		EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton"..i, 1)
	end
	if mode == 3 then
		if EnhancedStackSplit_Options.XLmode then
			if maxStackSize > 1 and maxStackSize < 10 then
				for i = maxStackSize-1, 9 do
					EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton"..i, 0)
				end
			end
			if maxStackSize < 11 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton11", 0)
			end
			if maxStackSize < 13 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton12", 0)
			end
			if maxStackSize < 15 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton13", 0)
			end
			if maxStackSize < 16 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton14", 0)
			end
			if maxStackSize < 17 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton15", 0)
			end
			if maxStackSize < 21 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton16", 0)
			end
			if maxStackSize < 101 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton9", 0)
			end
		else
			if maxStackSize > 1 and maxStackSize < 11 then
				for i = maxStackSize-1, 10 do
					EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton"..i, 0)
				end
			end
			if maxStackSize < 21 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton10", 0)
			end
		end
		EnhancedStackSplit:ButtonTweak("EnhancedStackSplitAutoSplitButton", 0)
		StackSplitOkayButton:Hide()
		StackSplitLeftButton:Hide()
		StackSplitRightButton:Hide()
		EnhancedStackSplitAutoTextureFrame:Show()
		EnhancedStackSplitAuto1Button:Show()
		EnhancedStackSplitTextAutoText1:Show()
		EnhancedStackSplitTextAutoText2:Show()
		EnhancedStackSplitTextAutoText1:SetText("|cffffffff"..L["Free Bag Slots"]..":|r |cffff8000"..GetFreeBagSlots().."|r")
		EnhancedStackSplitTextAutoText2:SetText("")
	else
		if EnhancedStackSplit_Options.XLmode then
			if maxStackSize > 1 and maxStackSize < 10 then
				for i = maxStackSize, 9 do
					EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton"..i, 0)
				end
			end
			if maxStackSize < 10 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton11", 0)
			end
			if maxStackSize < 12 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton12", 0)
			end
			if maxStackSize < 14 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton13", 0)
			end
			if maxStackSize < 15 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton14", 0)
			end
			if maxStackSize < 16 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton15", 0)
			end
			if maxStackSize < 20 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton16", 0)
			end
			if maxStackSize < 100 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton9", 0)
			end
		else
			if maxStackSize > 1 and maxStackSize < 11 then
				for i = maxStackSize, 10 do
					EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton"..i, 0)
				end
			end
			if maxStackSize < 20 then
				EnhancedStackSplit:ButtonTweak("EnhancedStackSplitButton10", 0)
			end
		end
		if not autoSplitItemLink then
			EnhancedStackSplit:ButtonTweak("EnhancedStackSplitAutoSplitButton", 0)
		else
			EnhancedStackSplit:ButtonTweak("EnhancedStackSplitAutoSplitButton", 1)
		end
		EnhancedStackSplitAutoTextureFrame:Hide()
		EnhancedStackSplitAuto1Button:Hide()
		EnhancedStackSplitTextAutoText1:Hide()
		EnhancedStackSplitTextAutoText2:Hide()
		StackSplitOkayButton:Show()
		StackSplitLeftButton:Show()
		StackSplitRightButton:Show()
	end
	EnhancedStackSplitTextFrameTXT:SetText(splitMode[mode])
end

function EnhancedStackSplit:AutoSplitInfo(num)
	local freeSlots, numStacks, leftOver = AutoSplitCalc(num)
	EnhancedStackSplitTextAutoText2:SetText("|cffffffff"..maxStackSize.." | |cffff8000"..numStacks.."|cffffffffx "..num.." | "..L["leftover"].."="..leftOver.."|r")
end

function EnhancedStackSplit_OpenStackSplitFrame(maxStack, parent, anchor, anchorTo)
	if maxStack < 2 then return end

	PositionAnchor = anchor
	PositionParent = parent
	PositionAnchorTo = anchorTo

	splitItemBag = StackSplitFrame.owner:GetParent():GetID()
	splitItemSlot = StackSplitFrame.owner:GetID()

	local splitItemName = StackSplitFrame.owner:GetParent():GetName()
	if splitItemName then
		for i = 1, #mainBankFrames do -- this needs some better solution
			if string.find(splitItemName, mainBankFrames[i]) then
				splitItemBag = -1
			end
		end
		for i = 1, #blockedFrames do -- this needs some better solution
			if string.find(splitItemName, blockedFrames[i]) then
				splitItemBag = nil
			end
		end
	end

	autoSplitItemLink = nil
	if splitItemBag then
		autoSplitItemLink = GetContainerItemLink(splitItemBag, splitItemSlot)
	end

	maxStackSize = maxStack
	EnhancedStackSplit:PositionSplitFrame()
	StackSplitOkayButton:SetPoint("RIGHT", "StackSplitFrame", "BOTTOM", -3, 40)
	StackSplitCancelButton:SetPoint("LEFT", "StackSplitFrame", "BOTTOM", 5, 40)
	autoSplitMode = false
	EnhancedStackSplit:AutoSplitButtonToggle(true)
	EnhancedStackSplitTextFrameTXT:SetText(splitMode[EnhancedStackSplit_Options.mode])
	EnhancedStackSplit:ModeSettings(EnhancedStackSplit_Options.mode)
end

function EnhancedStackSplit:Split(num)
	if autoSplitMode then
		local freeSlots, numStacks, leftOver = AutoSplitCalc(num)
		autoSplitStackCount = 0
		autoSplitNumStacks = numStacks
		autoSplitLeftOver = leftOver
		autoSplitNumber = num
		ClearSlotLocks()
		EnhancedStackSplit:AutoSplitButtonToggle(false)
		EnhancedStackSplit:AutoSplit(autoSplitNumStacks, autoSplitNumber, autoSplitLeftOver)
	else
		EnhancedStackSplit:SingleSplit(num)
	end
end

function EnhancedStackSplit:CheckItemLock(arg1, arg2)
	if StackSplitFrame:IsShown() and StackSplitFrame.owner and splitItemBag and splitItemSlot then
		if arg1 and arg2 then
			if arg1 == splitItemBag and arg2 == splitItemSlot then
				local locked = select(3, GetContainerItemInfo(splitItemBag, splitItemSlot))
				if not locked then
					EnhancedStackSplit:AutoSplit(autoSplitNumStacks, autoSplitNumber, autoSplitLeftOver)
				end
			end
		end
	end
end

function EnhancedStackSplit:AutoSplit(numStacks, num, leftOver)
	if leftOver == 0 then
		if autoSplitStackCount == numStacks-1 then
			StackSplitFrame:Hide() return
		end
	else
		if autoSplitStackCount == numStacks then
			StackSplitFrame:Hide() return
		end
	end
	local bag, slot = GetFreeSlot()
	SplitContainerItem(splitItemBag, splitItemSlot, num)
	if bag ~= nil then
		LockSlot(bag, slot)
		PickupContainerItem(bag, slot)
	end
	autoSplitStackCount = autoSplitStackCount + 1
end

function EnhancedStackSplit:SingleSplit(num)
	if num >= StackSplitFrame.maxStack then
		num = StackSplitFrame.maxStack
		StackSplitRightButton:Disable()
	end
	if num < StackSplitFrame.maxStack then
		StackSplitRightButton:Enable()
	end
	StackSplitLeftButton:Enable()
	StackSplitFrame.split = num
	StackSplitText:SetText(num)
	if EnhancedStackSplit_Options.mode == 2 then
		StackSplitFrameOkay_Click()
	end
end
-- ---------------------------------------------------------------------------------------------------------------------

-- ---------------------------------------------------------------------------------------------------------------------
local function OnEvent(self, event, ...)
	if event == "ITEM_LOCK_CHANGED" then
		EnhancedStackSplit:CheckItemLock(arg1, arg2)
	elseif event == "ADDON_LOADED" then
		if arg1 == "EnhancedStackSplit" then
			EnhancedStackSplit:InitOptions()
			EnhancedStackSplit:CreateFrames()
			hooksecurefunc("OpenStackSplitFrame", EnhancedStackSplit_OpenStackSplitFrame)
			EnhancedStackSplit:UnregisterEvent("ADDON_LOADED")
		end
	end
end
-- ---------------------------------------------------------------------------------------------------------------------

EnhancedStackSplit:RegisterEvent("ADDON_LOADED")
EnhancedStackSplit:RegisterEvent("ITEM_LOCK_CHANGED")
EnhancedStackSplit:SetScript("OnEvent", OnEvent)