local _, BigBrother = ...

local module = BigBrother:NewModule("BuffCheck", "AceEvent-3.0")
local L = LibStub("AceLocale-3.0"):GetLocale("BigBrother")

local _G = _G

local CLASS_COLORS = _G.RAID_CLASS_COLORS

local DEFAULT_ROWS = 10
local DEFAULT_ROWS_MIN = 5
local DEFAULT_ROWS_MAX = 25

local BAR_HEIGHT = 18
local BAR_WIDTH = 200
local BUFF_SPACING = 18
local TOTAL_BUFFS = 5

local BuffWindow, BuffTable
local PlayersShown, RowsCreated
local SelectedBuff = 1
local PlayerList = {}
local DirtyBuffs

local CreateBuffRow, CreateBuffWindow, ResizeBuffWindow, UpdateBuffWindow, UpdateBuffWindowBuffs

do --BuffTable
	local function spellData(spellId)
		local name, _, icon = GetSpellInfo(spellId)
		return { name, icon }
	end

	--Load up local tables from master spell ID tables
	local function createBuffGroup(tbl)
		local t = {}
		for i,v in ipairs(tbl) do
			t[i] = spellData(v)
		end
		return t
	end

	local function buffSort(a, b)
		if a.buffs < b.buffs then
			return true
		elseif a.buffs > b.buffs then
			return false
		elseif a.name < b.name then
			return true
		end
		return false
	end

	local function classSort(a, b)
		if a.class < b.class then
			return true
		elseif a.class > b.class then
			return false
		elseif a.name < b.name then
			return true
		end
		return false
	end

	BuffTable = {
		{
			name = L["Raid Buffs"],
			sortFunc = buffSort,
			buffs = {
				-- 1459 Arcane Intellect, 23028 Arcane Brilliance, 61024 Dalaran Intellect, 61316 Dalaran Brilliance
				{spellData(1459), spellData(23028),	spellData(61024),	spellData(61316)},
				-- 1243 Power Word: Fortitude, 21562 Prayer of Fortitude, 69377 Fortitude (scroll)
				{spellData(1243), spellData(21562), spellData(69377)},
				-- 1126 Mark of the Wild, 21849 Gift of the Wild, 69381 Gift of the Wild (drums)
				{spellData(1126), spellData(21849), spellData(69381)},
				-- 14752 Divine Spirit, 27681 Prayer of Spirit
				{spellData(14752), spellData(27681)},
				-- 976 Shadow Protection, 27683 Prayer of Shadow Protection, 19876 Shadow Resistance Aura 
				{spellData(976), spellData(27683), spellData(19876)},
			}
		},
		{
			name = L["Paladin Buffs"],
			sortFunc = classSort,
			buffs = {
				-- 20217 Blessing of Kings, 25898 Greater Blessing of Kings, 69378 Blessing of Forgotten Kings (drums)
				{spellData(20217), spellData(25898), spellData(69378)}, 
				-- 19740 Blessing of Might, 25782 Greater Blessing of Might, 6673 Battle Shout
				{spellData(19740), spellData(25782), spellData(6673)},
				-- 19742 Blessing of Wisdom, 25894 Greater Blessing of Wisdom, 5677 Mana Spring Totem
				{spellData(19742), spellData(25894), spellData(5677)},
				-- 20911 Blessing of Sanctuary, 25899 Greater Blessing of Sanctuary, 63944 Renewed Hope
				{spellData(20911), spellData(25899), spellData(63944)},
			}
		},
		{
			name = L["Consumables"],
			sortFunc = classSort,
			buffs = {
				{},
				createBuffGroup(BigBrother.SpellData.flasks),
				createBuffGroup(BigBrother.SpellData.elixirsBattle),
				createBuffGroup(BigBrother.SpellData.elixirsGuardian),
				createBuffGroup(BigBrother.SpellData.foods),
			}
		}
	}
end


local function GetHeightFromRows(rows)
	return 4 + BUFF_SPACING + 4 + ((BAR_HEIGHT + 2) * rows) + 4 + 0.5
end

local function GetRowsFromHeight(height)
	return math.floor( (height - (BUFF_SPACING + 4) - 8) / (BUFF_SPACING + 2) )
end

local function SetDirty()
	DirtyBuffs = true
end

local function BuffWindow_OnEnter(frame)
	GameTooltip:SetOwner(frame, "ANCHOR_TOPLEFT")
	if frame.BuffName then
		GameTooltip:SetUnitBuff(frame.unit, frame.BuffName)
	elseif frame.tooltip then
		GameTooltip:AddLine(frame.tooltip)
	end
	GameTooltip:Show()
end

local function BuffWindow_OnLeave()
	GameTooltip:Hide()
end

local BuffWindow_OnUpdate
do --BuffWindow_OnUpdate
	local total = 0
	function BuffWindow_OnUpdate(_, elapsed)
		total = total + elapsed
		if total >= 0.5 and DirtyBuffs then
			UpdateBuffWindowBuffs()
			total = 0
			DirtyBuffs = nil
		end
	end
end

function CreateBuffRow(parent, xOffset, yOffset)
	local row = CreateFrame("Frame", nil, parent)
	row:SetPoint("TOPLEFT", parent, "TOPLEFT", xOffset, yOffset)
	row:SetHeight(BAR_HEIGHT)
	row:SetWidth(BAR_WIDTH)
	
	row.Background = row:CreateTexture(nil, "BACKGROUND")
	row.Background:SetAllPoints(row)
	row.Background:SetTexture("Interface\\Buttons\\WHITE8X8.blp")
	row.Background:SetGradientAlpha("HORIZONTAL", 0.5, 0.0, 0.0, 0.8, 0.5, 0.0, 0.0, 0.0)

	row.Name = row:CreateFontString(nil, "OVERLAY", "GameFontNormal")	
	row.Name:SetPoint("LEFT", row, "LEFT", 4, 0)
	row.Name:SetTextColor(1.0, 1.0, 1.0)
	row.Name:SetText("Test")
	
	row.Buff = {}
	for i = 1, TOTAL_BUFFS do
		row.Buff[i] = CreateFrame("Frame", nil, row)
		row.Buff[i]:SetPoint("RIGHT", row, "RIGHT", -4 - (TOTAL_BUFFS - i) * BUFF_SPACING, 0)
		row.Buff[i]:SetHeight(16)
		row.Buff[i]:SetWidth(16)
		
		row.Buff[i].texture = row.Buff[i]:CreateTexture(nil, "OVERLAY")
		row.Buff[i].texture:SetAllPoints(row.Buff[i])
		row.Buff[i].texture:SetTexture("Interface\\Buttons\\UI-CheckBox-Check.blp")
		
		row.Buff[i].BuffName = nil
		row.Buff[i]:SetScript("OnEnter", BuffWindow_OnEnter)
		row.Buff[i]:SetScript("OnLeave", BuffWindow_OnLeave)
		row.Buff[i]:EnableMouse()
	end

	return row
end

function CreateBuffWindow()
	local frame = CreateFrame("Frame", nil, UIParent)
	frame:Hide()
	
	frame:ClearAllPoints()
	if module.db.profile.x and module.db.profile.y then
		frame:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", module.db.profile.x, module.db.profile.y)
	else
		frame:SetPoint("CENTER", UIParent)
	end
	local height = module.db.profile.height or GetHeightFromRows(DEFAULT_ROWS)
	
	frame:SetBackdrop({
		bgFile = "Interface/Tooltips/UI-Tooltip-Background", 
		edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
		tile = true, tileSize = 16, edgeSize = 16, 
		insets = { left = 4, right = 4, top = 4, bottom = 4 }
	})
	frame:SetBackdropColor(0, 0, 0, 0.5)
	
	frame:EnableMouse()
	frame:SetMovable(true)
	frame:SetClampedToScreen(true)
	frame:SetResizable(true)
	frame:SetMinResize(BAR_WIDTH + 16 + 24, GetHeightFromRows(DEFAULT_ROWS_MIN))
	frame:SetMaxResize(BAR_WIDTH + 16 + 24, GetHeightFromRows(DEFAULT_ROWS_MAX))
	frame:SetWidth(BAR_WIDTH + 16 + 24)
	frame:SetHeight(height)
	
	frame:SetScript("OnMouseDown", function(self, button) 
		if button == "LeftButton" then
			self:StartMoving()
			self.isMoving = true
		end
	end)
	frame:SetScript("OnMouseUp", function(self) 
		if self.isMoving then
			self:StopMovingOrSizing()
			self.isMoving = false
			module.db.profile.x = self:GetLeft()
			module.db.profile.y = self:GetTop()
		end
	end)
	frame:SetScript("OnHide", function(self) 
		if self.isMoving then
			self:StopMovingOrSizing()
			self.isMoving = false
		end
	end)
	frame:SetScript("OnShow", UpdateBuffWindowBuffs)
	frame:SetScript("OnUpdate", BuffWindow_OnUpdate)
	frame:SetScript("OnSizeChanged", function(self)
		if self.isResizing then
			ResizeBuffWindow()
		end
	end)
	
	local title = frame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
	title:SetPoint("TOP", frame, "TOP", 0, -8)
	title:SetTextColor(1.0, 1.0, 1.0)
	
	local leftButton = CreateFrame("Button", nil, frame)
	leftButton:SetNormalTexture("Interface\\Buttons\\UI-SpellbookIcon-PrevPage-Up.blp")
	leftButton:SetPushedTexture("Interface\\Buttons\\UI-SpellbookIcon-PrevPage-Down.blp")
	leftButton:SetWidth(16)
	leftButton:SetHeight(18)
	leftButton:SetPoint("TOPLEFT", frame, "TOPLEFT", 64, -5)
	leftButton:SetScript("OnClick", function() 
		SelectedBuff = SelectedBuff - 1
		if SelectedBuff == 0 then
			SelectedBuff = #BuffTable
		end
		UpdateBuffWindowBuffs()
	end)
	
	local rightButton = CreateFrame("Button", nil, frame)
	rightButton:SetNormalTexture("Interface\\Buttons\\UI-SpellbookIcon-NextPage-Up.blp")
	rightButton:SetPushedTexture("Interface\\Buttons\\UI-SpellbookIcon-NextPage-Down.blp")
	rightButton:SetWidth(16)
	rightButton:SetHeight(18)
	rightButton:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -64, -5)
	rightButton:SetScript("OnClick", function() 
		SelectedBuff = SelectedBuff + 1
		if SelectedBuff > #BuffTable then
			SelectedBuff = 1
		end
		UpdateBuffWindowBuffs()
	end)
	
	local closeButton = CreateFrame("Button", nil, frame)
	closeButton:SetNormalTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Up.blp")
	closeButton:SetPushedTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Down.blp")
	closeButton:SetHighlightTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Highlight.blp")
	closeButton:SetWidth(22)
	closeButton:SetHeight(22)
	closeButton:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -4, -4)
	closeButton:SetScript("OnClick", function(self) self:GetParent():Hide() end)
	
	local readyButton = CreateFrame("Button", nil, frame)
	readyButton:SetNormalTexture("Interface\\RAIDFRAME\\ReadyCheck-Waiting")
	readyButton:SetWidth(12)
	readyButton:SetHeight(12)
	readyButton:SetPoint("TOPLEFT", frame, "TOPLEFT", 8, -8)
	readyButton:SetScript("OnClick", function() DoReadyCheck() end)
	readyButton:SetScript("OnEnter", BuffWindow_OnEnter)
	readyButton:SetScript("OnLeave", BuffWindow_OnLeave)
	readyButton.tooltip = "Ready Check"
	
	local checkButton = CreateFrame("Button", nil, frame)
	checkButton:SetNormalTexture("Interface\\RAIDFRAME\\ReadyCheck-Ready")
	checkButton:SetWidth(12)
	checkButton:SetHeight(12)
	checkButton:SetPoint("TOPLEFT", frame, "TOPLEFT", 22, -8)
	checkButton:SetScript("OnClick", function() BigBrother:ConsumableCheck("GROUP") end)
	checkButton:SetScript("OnEnter", BuffWindow_OnEnter)
	checkButton:SetScript("OnLeave", BuffWindow_OnLeave)
	checkButton.tooltip = "Quick Check"

	local rows = {}
	for i = 1, GetRowsFromHeight(height) do
		rows[i] = CreateBuffRow(frame, 8, -4 - i * (BUFF_SPACING + 2))
	end
	RowsCreated = #rows
	PlayersShown = #rows
	
	local scrollBar = CreateFrame("ScrollFrame", "BigBrother_BuffCheck_ScrollFrame", frame, "FauxScrollFrameTemplate")
	scrollBar:SetScript("OnVerticalScroll", function(self, offset)
		FauxScrollFrame_OnVerticalScroll(self, offset, BAR_HEIGHT + 2, UpdateBuffWindow)
	end)
	scrollBar:SetPoint("TOPLEFT", rows[1], "TOPLEFT", 0, 0)
	scrollBar:SetPoint("BOTTOMRIGHT", rows[#rows], "BOTTOMRIGHT", 0, 0)

	-- drag handle
	local dragHandle = CreateFrame("Frame", nil, frame)
	dragHandle:Show()
	dragHandle:SetFrameLevel(frame:GetFrameLevel() + 10) -- place this above everything
	dragHandle:SetWidth(BAR_WIDTH + 16 + 24)
	dragHandle:SetHeight(8)
	dragHandle:SetPoint("BOTTOM", frame, "BOTTOM", 0, 0)
	dragHandle:EnableMouse(true)
	dragHandle:SetBackdrop({
		bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
		tile = true, tileSize = 16, edgeSize = 0,
		insets = { left = 6, right = 6, top = 3, bottom = 3 }
	})
	dragHandle:SetBackdropColor(1, 1, 1, 0)
	dragHandle:SetScript("OnEnter", function(self)
		self:SetBackdropColor(1, 1, 1, 0.8)
	end)
	dragHandle:SetScript("OnLeave", function(self)
		self:SetBackdropColor(1, 1, 1, 0)
	end)
	dragHandle:SetScript("OnMouseDown", function(self, button)
		if button == "LeftButton" then
			self:GetParent().isResizing = true
			self:GetParent():StartSizing("BOTTOMRIGHT")
		end
	end)
	dragHandle:SetScript("OnMouseUp", function(self, button)
		if button == "LeftButton" then
			self:GetParent():StopMovingOrSizing()
			self:GetParent().isResizing = false
			self:GetParent():SetHeight(GetHeightFromRows(PlayersShown))
		elseif button == "RightButton" then
			self:GetParent():SetHeight(GetHeightFromRows(DEFAULT_ROWS))
		end
		ResizeBuffWindow()
	end)
	
	frame.Title = title
	frame.LeftButton = leftButton
	frame.RightButton = rightButton
	frame.CloseButton = closeButton
	frame.ReadyButton = readyButton
	frame.CheckButton = checkButton
	frame.Rows = rows
	frame.ScrollBar = scrollBar
	frame.DragHandle = dragHandle
	
	BuffWindow = frame
end

--update the player buff table
do --UpdateBuffWindowBuffs
	local partyUnits = {"player"}
	for i = 1, MAX_PARTY_MEMBERS do
		partyUnits[#partyUnits + 1] = ("party%d"):format(i)
	end
	local raidUnits = {}
	for i = 1, MAX_RAID_MEMBERS do
		raidUnits[#raidUnits + 1] = ("raid%d"):format(i)
	end
	
	function UpdateBuffWindowBuffs()
		local groups = BigBrother:GetCheckedGroups()
		local currentZone = GetRealZoneText()
		
		local inRaid = GetNumRaidMembers() > 0
		
		local index = 1
		for i, unit in ipairs(inRaid and raidUnits or partyUnits) do
			local name, subgroup, class, zone, online, dead
			if inRaid then
				name, _, subgroup, _, _, class, zone, online, dead = GetRaidRosterInfo(i)
			else
				name = UnitName(unit)
				subgroup = 1
				class = select(2, UnitClass(unit)) or _G.UNKNOWN --cross-realm players take a sec to load, don't break sorting prz
				online = UnitIsConnected(unit)
				dead = UnitIsDeadOrGhost(unit)
			end
			if not name then break end
			
			if groups[subgroup] then
				local player = PlayerList[index]
				if not player then
					player = { buff = {} }
				end
				PlayerList[index] = player
				
				player.name = name
				player.class = class
				player.status = not online or dead or (zone and zone ~= currentZone)
				player.unit = unit
				player.buffs = 0
				wipe(player.buff)
				
				for i, buffGroup in ipairs(BuffTable[SelectedBuff].buffs) do
					for _, buffs in ipairs(buffGroup) do
						if UnitBuff(player.unit, buffs[1]) then
							if module.db.profile.shortBuffs or select(6, UnitBuff(player.unit, buffs[1])) > 360 then
								player.buff[i] = buffs
								player.buffs = player.buffs + 1
							end
							break
						end
					end
				end
				
				index = index + 1
			end
		end
		
		-- clear the rest of the table so we dont get nil holes that lead to ill-defined behavior in sort
		while PlayerList[index] do
			PlayerList[index] = nil
			index = index + 1
		end
		sort(PlayerList, BuffTable[SelectedBuff].sortFunc)
		
		UpdateBuffWindow()
	end
end

--update the display
function UpdateBuffWindow()
	BuffWindow.Title:SetText(BuffTable[SelectedBuff].name)
	
	FauxScrollFrame_Update(BuffWindow.ScrollBar, #PlayerList, PlayersShown, 20)
	local offset = FauxScrollFrame_GetOffset(BuffWindow.ScrollBar)
	
	for i = 1, PlayersShown do
		if PlayerList[i + offset] then
			local player = PlayerList[i + offset]
			local color = not player.status and _G.HIGHLIGHT_FONT_COLOR or _G.RED_FONT_COLOR
			local bgColor = CLASS_COLORS[player.class] or GRAY_FONT_COLOR
			BuffWindow.Rows[i].Name:SetText(player.name)
			BuffWindow.Rows[i].Name:SetTextColor(color.r, color.g, color.b)
			BuffWindow.Rows[i].Background:SetGradientAlpha(
				"HORIZONTAL",
				bgColor.r/1.5, bgColor.g/1.5, bgColor.b/1.5, 0.8,
				bgColor.r/1.5, bgColor.g/2, bgColor.b/1.5, 0
			)

			for j = 1, 5 do
				if player.buff[j] then
					BuffWindow.Rows[i].Buff[j].BuffName = player.buff[j][1]
					BuffWindow.Rows[i].Buff[j].texture:SetTexture(player.buff[j][2])
					BuffWindow.Rows[i].Buff[j].unit = player.unit
					BuffWindow.Rows[i].Buff[j]:Show()
				else
					BuffWindow.Rows[i].Buff[j]:Hide()
				end
			end
			BuffWindow.Rows[i]:Show()
		else
			BuffWindow.Rows[i]:Hide()
		end
	end
end

function ResizeBuffWindow()
	--clamp the list size
	if module.db.profile.limitResize then
		local minSize, maxSize = GetHeightFromRows(DEFAULT_ROWS_MIN), GetHeightFromRows(#PlayerList)
		if BuffWindow:GetHeight() > maxSize then
			BuffWindow:SetHeight(maxSize < minSize and minSize or maxSize)
		end
	end
	
	local numVisibleRows = GetRowsFromHeight(BuffWindow:GetHeight())

	if numVisibleRows > RowsCreated then
		for i = (1 +  RowsCreated), numVisibleRows do
			BuffWindow.Rows[i] = CreateBuffRow(BuffWindow, 8, -4 - i * (BUFF_SPACING + 2))
			BuffWindow.Rows[i]:Hide()
		end
		RowsCreated = numVisibleRows
	end

	if numVisibleRows < PlayersShown then
		for i = (1 + numVisibleRows), PlayersShown do
			BuffWindow.Rows[i]:Hide()
		end
	end
	PlayersShown = numVisibleRows
	
	BuffWindow.ScrollBar:SetPoint("BOTTOMRIGHT", BuffWindow.Rows[PlayersShown], "BOTTOMRIGHT", 0, 0)
	UpdateBuffWindow()
	
	module.db.profile.height = BuffWindow:GetHeight()
end


function module:OnInitialize()
	self.db = BigBrother.db:RegisterNamespace("BuffCheck", {
		profile = {
			limitResize = false,
			shortBuffs = true,
		}
	})
end

function module:OnEnable()
	self:RegisterEvent("UNIT_AURA", SetDirty)
	self:RegisterEvent("RAID_ROSTER_UPDATE", SetDirty)
	self:RegisterEvent("PARTY_MEMBERS_CHANGED", SetDirty)
	
	if _G.CUSTOM_CLASS_COLORS then
		local function updateClassColors()
			for k,v in next, _G.CUSTOM_CLASS_COLORS do
				CLASS_COLORS[k] = v
			end
		end
		_G.CUSTOM_CLASS_COLORS:RegisterCallback(updateClassColors)
		updateClassColors()
	end
end

function module:ToggleWindow()
	if BuffWindow and BuffWindow:IsShown() then
		BuffWindow:Hide()
	elseif BuffWindow then
		BuffWindow:Show()
	else
		CreateBuffWindow()
		BuffWindow:Show()
	end
end

function module:AddOptions(options)
	options.args.buffcheck = {
		name = L["Buff Check"],
		desc = L["Pops up a window to check raid buffs (drag the bottom to resize)."],
		type = "execute",
		func = self.ToggleWindow,
		order = 8,
		width = "full",
	}
	
	options.args.buffwindow = {
		name = L["Buff Check"],
		--desc = L["Buff Check"],
		type = 'group',
		get = function(info) return self.db.profile[info[#info]] end,
		set = function(info, value)
			self.db.profile[info[#info]] = value
			SetDirty()
		end,
		order = 30,
		args = {
			limitResize = {
				name = L["Limit size"],
				desc = (L["Limits the number of rows shown to the number of people in your group (minimum: %d, maximum: %d)"]):format(DEFAULT_ROWS_MIN, DEFAULT_ROWS_MAX),
				type = 'toggle',
				order = 10,
			},
			shortBuffs = {
				name = L["Show temporal buffs"],
				desc = L["Include short duration buffs, totems, and auras when checking buffs"],
				type = 'toggle',
				order = 10,
			},
		}
	}
end
