local _, BigBrother = ...

local module = BigBrother:NewModule("Alerts", "AceEvent-3.0")
local L = LibStub("AceLocale-3.0"):GetLocale("BigBrother")

local _G = _G
local bit_band = _G.bit.band

local UNKNOWN = ("(%s)"):format(_G.UNKNOWN)

--localize names for reverse lookup
local crowdControlSpells = {}
for _,v in ipairs(BigBrother.SpellData.crowdControl) do
	crowdControlSpells[(GetSpellInfo(v))] = true
end

local misdirectSpells = {}
for _,v in ipairs(BigBrother.SpellData.misdirect) do
	misdirectSpells[v] = true
end

local tauntSpells = {}
local tauntAoESpells = {}
for _,v in ipairs(BigBrother.SpellData.taunt) do
	tauntSpells[(GetSpellInfo(v))] = true
end
for _,v in ipairs(BigBrother.SpellData.tauntAoE) do
	tauntAoESpells[(GetSpellInfo(v))] = true
end

local interruptSpells = {}
for _,v in ipairs(BigBrother.SpellData.interrupts) do
	interruptSpells[(GetSpellInfo(v))] = true
end

local combatResSpells = {}
for _, v in ipairs(BigBrother.SpellData.combatRes) do
	combatResSpells[(GetSpellInfo(v))] = true
end


function module:OnInitialize()
	self.db = BigBrother.db:RegisterNamespace("Alerts", {
		profile = {
			crowdControl = true,
			misdirect = true,
			taunt = false,
			interrupt = false,
			combatRes = true,
			spellLink = true,
			groupOnly = true,
			outputSelf = true,
			outputParty = false,
			outputRaid = false,
			outputGuild = false,
			outputOfficer = false,
		}
	})
end

function module:OnEnable()
	self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
end


local function GetIconString(flags)
	local raidTarget = bit_band(flags, _G.COMBATLOG_OBJECT_RAIDTARGET_MASK)
	if raidTarget ~= 0 then
		for i=1,8 do
			local mask = _G.COMBATLOG_OBJECT_RAIDTARGET1 * (2 ^ (i - 1)) --_G["COMBATLOG_OBJECT_RAIDTARGET"..i]
 			if raidTarget == mask then
				return _G.TEXT_MODE_A_STRING_DEST_ICON:format(mask, _G["COMBATLOG_ICON_RAIDTARGET"..i])
			end
		end
	end
	return ""
end


function module:Spam(msg)
	if not msg or msg == "" then return end
	local chatMsg = string.gsub(msg, "|Hicon:%d+:dest|h|TInterface.TargetingFrame.UI%-RaidTargetingIcon_(%d)%.blp:0|t|h", "{rt%1}")
	
	if self.db.profile.outputSelf then
		BigBrother:Print(msg)
	end
	if self.db.profile.outputGroup then
		if GetNumRaidMembers() > 0 then
			SendChatMessage(chatMsg, "RAID")
		elseif GetNumPartyMembers() > 0 then
			SendChatMessage(chatMsg, "PARTY")
		end
	else
		if self.db.profile.outputRaid then
			SendChatMessage(chatMsg, "RAID")
		end
		if self.db.profile.outputParty then
			SendChatMessage(chatMsg, "PARTY")
		end
	end
	
	local checkGuild = IsInGuild() and (GetNumPartyMembers() > 0 or GetNumRaidMembers() > 0)
	if self.db.profile.outputGuild and checkGuild then
		SendChatMessage(chatMsg, "GUILD")
	end
	if self.db.profile.outputOfficer and checkGuild then
		SendChatMessage(chatMsg, "OFFICER")
	end
end

do
	local spamCount = 0
	local spamLastTime = 0
	
	local function checkSrc(srcName, srcFlags)
		if module.db.profile.groupOnly then
			return srcName and UnitName(srcName) and true or false
		end
		return true
	end
	
	function module:COMBAT_LOG_EVENT_UNFILTERED(_, _, event, _, srcName, srcFlags, _, dstName, dstFlags, spellId, spellName, _, ...)
		
		--CC Breaking
		if self.db.profile.crowdControl
			and (event == "SPELL_AURA_BROKEN" or event == "SPELL_AURA_BROKEN_SPELL"
				or (event == "SPELL_AURA_REMOVED" and (spellId == 51514))) --Hex
			and crowdControlSpells[spellName]
			and checkSrc(srcName)
			and (bit_band(dstFlags, _G.COMBATLOG_OBJECT_REACTION_HOSTILE) ~= 0) then
			
			local srcOutput = ("%s|cff40ff40%s|r"):format(GetIconString(srcFlags), srcName or UNKNOWN)
			local dstOutput = ("%s|cffff4040%s|r"):format(GetIconString(dstFlags), dstName or UNKNOWN)
			local spellOutput = self.db.profile.spellLink and GetSpellLink(spellId) or spellName
			
			local output
			if event == "SPELL_AURA_BROKEN" or event == "SPELL_AURA_REMOVED" then
				output = (L["%s on %s removed by %s"]):format(spellOutput, dstOutput, srcOutput)
			elseif event == "SPELL_AURA_BROKEN_SPELL" then
				local extraSpellId, extraSpellName = ...
				local extraOutput = self.db.profile.spellLink and GetSpellLink(extraSpellId) or extraSpellName
				output = (L["%s on %s removed by %s's %s"]):format(spellOutput, dstOutput, srcOutput, extraOutput)
			end
			
			--throttling...hmmm
			local now = GetTime()
			if now - spamLastTime > 3 then
				spamCount = 0
			end
			spamCount = spamCount + 1
			if spamCount > 3 then
				output = nil --("%s is being bad"):format(srcOut) --
			end
			spamLastTime = now
			
			self:Spam(output)
			return
		end
		
		--Misdirects
		if self.db.profile.misdirect and event == "SPELL_CAST_SUCCESS" and misdirectSpells[spellId] and checkSrc(srcName) then
			local srcOutput = ("%s|cff40ff40%s|r"):format(GetIconString(srcFlags), srcName or UNKNOWN)
			local dstOutput = ("%s|cffff4040%s|r"):format(GetIconString(dstFlags), dstName or UNKNOWN)
			local spellOutput = self.db.profile.spellLink and GetSpellLink(spellId) or spellName
			
			self:Spam(L["%s cast %s on %s"]:format(srcOutput, spellOutput, dstOutput))
			return
		end
		
		--Interrupts
		if self.db.profile.interrupt and (event == "SPELL_INTERRUPT" or (event == "SPELL_MISSED" and interruptSpells[spellName])) and checkSrc(srcName) then
			local srcOutput = ("%s|cff40ff40%s|r"):format(GetIconString(srcFlags), srcName or UNKNOWN)
			local dstOutput = ("%s|cffff4040%s|r"):format(GetIconString(dstFlags), dstName or UNKNOWN)
			local spellOutput = self.db.profile.spellLink and GetSpellLink(spellId) or spellName
			
			local output
			if event == "SPELL_INTERRUPT" then --doesn't trigger on silence/stun :\ oh well
				local extraSpellId, extraSpellName = ... --the interrupted spell
				local extraSpellOutput = self.db.profile.spellLink and GetSpellLink(extraSpellId) or extraSpellName
				output = L["%s interrupted %s's %s with %s"]:format(srcOutput, dstOutput, extraSpellOutput, spellOutput)
			elseif event == "SPELL_MISSED" then
				local missType = ...
				output = L["%s cast %s on %s (%s)"]:format(srcOutput, spellOutput, dstOutput, _G["ACTION_SPELL_MISSED_"..missType])
			end
			
			self:Spam(output)
			return
		end
		
		--Taunts
		if self.db.profile.taunt and (
			((event == "SPELL_AURA_APPLIED" and tauntSpells[spellName]) or
					(event == "SPELL_MISSED" and tauntSpells[spellName]) or
					(event == "SPELL_CAST_SUCCESS" and tauntAoESpells[spellName]))
				and checkSrc(srcName)
			) then
			
			if self.debug then
				self.db.profile.debug = self.db.profile.debug or {}
				self.db.profile.debug[spellId] = {spellName, event, (...)}
			end
			
			local srcOutput = ("%s|cff40ff40%s|r"):format(GetIconString(srcFlags), srcName or UNKNOWN)
			local dstOutput = ("%s|cffff4040%s|r"):format(GetIconString(dstFlags), dstName or UNKNOWN)
			local spellOutput = self.db.profile.spellLink and GetSpellLink(spellId) or spellName
			
			local output
			if event == "SPELL_AURA_APPLIED" then
				output = L["%s cast %s on %s"]:format(srcOutput, spellOutput, dstOutput)
			elseif event == "SPELL_MISSED" then
				local missType = ...
				if spellName == (GetSpellInfo(49576)) and missType == "IMMUNE" then return end --ignore dg yoink immunes
				output = L["%s cast %s on %s (%s)"]:format(srcOutput, spellOutput, dstOutput, _G["ACTION_SPELL_MISSED_"..missType])
			elseif event == "SPELL_CAST_SUCCESS" then
				output = L["%s cast %s"]:format(srcOutput, spellOutput)
			end
			
			self:Spam(output)
			return
		end
		
		--Battle Res
		if self.db.profile.combatRes and event == "SPELL_RESURRECT" and combatResSpells[spellId] and checkSrc(srcName) then
			local srcOutput = ("%s|cff40ff40%s|r"):format(GetIconString(srcFlags), srcName or UNKNOWN)
			local dstOutput = ("%s|cffff4040%s|r"):format(GetIconString(dstFlags), dstName or UNKNOWN)
			local spellOutput = self.db.profile.spellLink and GetSpellLink(spellId) or spellName
			
			self:Spam(L["%s cast %s on %s"]:format(srcOutput, spellOutput, dstOutput))
			return
		end
		
	end
end

function module:DoConfig(options)
	options.args.alerts = {
		name = L["Alerts"],
		--desc = L["Alerts"],
		type = 'group',
		get = function(info) return self.db.profile[info[#info]] end,
		set = function(info, value) self.db.profile[info[#info]] = value end,
		order = 20,
		args = {
			report = {
				name = L["Report"],
				desc = L["Set what is reported to chat"],
				type = 'group',
				inline = true,
				order = 10,
				args = {
					crowdControl = {
						name  = L["Crowd Control"],
						desc = L["Report when a player breaks a crowd control effect"],
						type = 'toggle',
						order = 5,
					},
					misdirect = {
						name  = L["Misdirect"],
						desc = L["Report who gains Misdirection"],
						type = 'toggle',
						order = 10,
					},
					taunt = {
						name  = L["Taunt"],
						desc = L["Report taunts"],
						type = 'toggle',
						order = 15,
					},
					interrupt = {
						name  = L["Interrupts"],
						desc = L["Report interrupts"],
						type = 'toggle',
						order = 16,
					},
					combatRes = {
						name = L["Combat Resurrection"],
						desc = L["Report battle resurrection casts"],
						type = 'toggle',
						order = 17,
					},
				},
			}, --report
			
			output = {
				name = L["Output"],
				desc = L["Set where the alert output is sent"],
				type = 'group',
				inline = true,
				order = 20,
				args = {
					outputSelf = {
						name = L["Self"],
						desc = L["Reports alerts to yourself"],
						type = 'toggle',
						order = 5,
					},
					outputParty = {
						name = L["Party"],
						desc = L["Reports alerts to party chat"],
						type = 'toggle',
						order = 11,
					},
					outputRaid = {
						name = L["Raid"],
						desc = L["Reports alerts to raid chat"],
						type = 'toggle',
						order = 12,
					},
					outputGuild = {
						name = L["Guild"],
						desc = L["Reports alerts to guild chat"],
						type = 'toggle',
						order = 15,
					},
					outputOfficer = {
						name = L["Officer"],
						desc = L["Reports alerts to officer chat"],
						type = 'toggle',
						order = 20,
					},
				},
			},
			
			spellLink = {
				name = L["Use Spell Links"],
				desc = L["Display spell names as clickable spell links"],
				type = 'toggle',
				order = 40,
			},
			groupOnly = {
				name = L["Group Only"],
				desc = L["Only report events from players in your party or raid group"],
				type = 'toggle',
				order = 41,
			},
			
		},
	}
end
